#include "crc_cache.h"

////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
//                                                                            //
// This file is distributed as part of the Cache Replacement Championship     //
// workshop held in conjunction with ISCA'2010.                               //
//                                                                            //
//                                                                            //
// Everyone is granted permission to copy, modify, and/or re-distribute       //
// this software.                                                             //
//                                                                            //
// Please contact Aamer Jaleel <ajaleel@gmail.com> should you have any        //
// questions                                                                  //
//                                                                            //
////////////////////////////////////////////////////////////////////////////////

// --- DO NOT EDIT THIS FILE --- DO NOT EDIT THIS FILE --- DO NOT EDIT THIS FILE ---
// --- DO NOT EDIT THIS FILE --- DO NOT EDIT THIS FILE --- DO NOT EDIT THIS FILE ---
// --- DO NOT EDIT THIS FILE --- DO NOT EDIT THIS FILE --- DO NOT EDIT THIS FILE ---

// IMPORTANT NOTE: DO NOT CHANGE ANYTHING IN THIS HEADER FILE. Changing anything
// in here will violate the competition rules.

string crc_access_names[] =
{
    "IFETCH   ",
    "LOAD     ",
    "STORE    ",
    "-NOP0-   ",
    "-NOP1-   ",
    "PREFETCH ",
    "WRITEBACK"
};

////////////////////////////////////////////////////////////////////////////////
//                                                                            //
// The constructor for the cache with appropriate cache parameters as args    //
//                                                                            //
////////////////////////////////////////////////////////////////////////////////
CRC_CACHE::CRC_CACHE( UINT32 _cacheSize, UINT32 _assoc, UINT32 _tpc, UINT32 _linesize, UINT32 _pol ) 
{

    // Start off with empty cache and replacement state
    cache          = NULL;
    cacheReplState = NULL;

    // Initialize parameters to the cache
    numsets  = _cacheSize / (_linesize * _assoc);
    assoc    = _assoc;
    threads  = _tpc;
    linesize = _linesize;

    replPolicy = _pol;

    // Initialize the cache
    InitCache();

    // Initialize Replacement State
    InitCacheReplacementState();

    // Initialize the stats
    InitStats();
}

////////////////////////////////////////////////////////////////////////////////
//                                                                            //
// The function initializes the cache hardware and structures                 //
//                                                                            //
////////////////////////////////////////////////////////////////////////////////
void CRC_CACHE::InitCache()
{
    // Initialize the Cache Access Functions
    lineShift  = CRC_FloorLog2( linesize );
    indexShift = CRC_FloorLog2( numsets );    
    indexMask  = (1 << indexShift) - 1;

    // Create the cache structure (first create the sets)
    cache = new LINE_STATE* [ numsets ];

    // ensure that we were able to create cache
    assert(cache);

    // If we were able to create the sets, now create the ways
    for(UINT32 setIndex=0; setIndex<numsets; setIndex++) 
    {
        cache[ setIndex ] = new LINE_STATE[ assoc ];

        // Initialize the cache ways
        for(UINT32 way=0; way<assoc; way++) 
        {
            cache[ setIndex ][ way ].tag   = 0xdeaddead;
            cache[ setIndex ][ way ].valid = false;
            cache[ setIndex ][ way ].dirty = false;
            cache[ setIndex ][ way ].sharing_dir   = 0;
        }
    }

    // Initialize cache access timer
    mytimer = 0;

}

////////////////////////////////////////////////////////////////////////////////
//                                                                            //
// The function initializes the stats for the cache                           //
//                                                                            //
////////////////////////////////////////////////////////////////////////////////
void CRC_CACHE::InitStats()
{
    for(UINT32 i=0; i<ACCESS_MAX; i++) 
    {
        lookups[i] = new COUNTER[ threads ];
        misses[i]  = new COUNTER[ threads ];
        hits[i]    = new COUNTER[ threads ];

        for(UINT32 t=0; t<threads; t++) 
        {
            lookups[i][t] = 0;
            misses[i][t]  = 0;
            hits[i][t]    = 0;
        }
    }
}

////////////////////////////////////////////////////////////////////////////////
//                                                                            //
// The function prints the statistics for the cache                           //
//                                                                            //
////////////////////////////////////////////////////////////////////////////////
ostream & CRC_CACHE::PrintStats(ostream &out)
{
    COUNTER totLookups = 0, totMisses = 0, totHits = 0;

    out<<"=========================================================="<<endl;
    out<<"==== Cache Replacement Championship -- LLC Statistics ===="<<endl;
    out<<"=========================================================="<<endl;
    out<<endl;
    out<<endl;    
    out<<"Cache Configuration: "<<endl;
    out<<"\tCache Size:     "<<(numsets*assoc*linesize/1024)<<"K"<<endl;
    out<<"\tLine Size:      "<<linesize<<"B"<<endl;
    out<<"\tAssociativity:  "<<assoc<<endl;
    out<<"\tTot # Sets:     "<<numsets<<endl;
    out<<"\tTot # Threads:  "<<threads<<endl;
    
    out<<endl;
    out<<"Cache Statistics: "<<endl;
    out<<endl;
    
    for(UINT32 a=0; a<ACCESS_MAX; a++) 
    {

        totLookups = 0;
        totMisses = 0;
        totHits = 0;

        for(UINT32 t=0; t<threads; t++) 
        {
            totLookups += lookups[a][t];
            totMisses  += misses[a][t];
            totHits    += hits[a][t];
        }

        if( totLookups ) 
        {
            out<<"\t"<<crc_access_names[a]<<" Accesses:   "<<totLookups<<endl;
            out<<"\t"<<crc_access_names[a]<<" Misses:     "<<totMisses<<endl;
            out<<"\t"<<crc_access_names[a]<<" Hits:       "<<totHits<<endl;
            out<<"\t"<<crc_access_names[a]<<" Miss Rate:  "<<((double)totMisses/(double)totLookups)*100.0<<endl;

            out<<endl;
        }
    }

    out<<endl;
    out<<"Per Thread Demand Reference Statistics: "<<endl;

    for(UINT32 t=0; t<threads; t++) 
    {
        totLookups = ThreadDemandLookupStats(t);
        totMisses  = ThreadDemandMissStats(t);
        totHits    = ThreadDemandHitStats(t);

        if( totLookups )
        {
            out<<"\tThread: "<<t<<" Lookups: "<<totLookups<<" Misses: "<<totMisses
                <<" Miss Rate: "<<((double)totMisses/(double)totLookups)*100.0<<endl;
        }
    }
    out<<endl;

    cacheReplState->PrintStats( out );
     
    return out;
}

////////////////////////////////////////////////////////////////////////////////
//                                                                            //
// The function slects a victim for the given set index. We enforce that      //
// we first fill all invalid entries. Once all invalid entries are filled     //
// the replacement policy is consulted to find the victim                     //
//                                                                            //
////////////////////////////////////////////////////////////////////////////////
INT32 CRC_CACHE::GetVictimInSet( UINT32 tid, UINT32 setIndex, Addr_t PC, Addr_t paddr, UINT32 accessType ) 
{
    // Get pointer to replacement state of current set
    LINE_STATE *vicSet = cache[ setIndex ];

    // First find and fill invalid lines
    for(UINT32 way=0; way<assoc; way++) 
    {
        if( vicSet[way].valid == false ) 
        {
            return way;
        }
    }

    // If no invalid lines, then replace based on replacement policy
    return cacheReplState->GetVictimInSet( tid, setIndex, vicSet, assoc, PC, paddr, accessType );
}

////////////////////////////////////////////////////////////////////////////////
//                                                                            //
// The function looks up the set for the tag and returns physical way index   //
// if the tag was a hit. Else returns -1 if it was a miss.                    //
//                                                                            //
////////////////////////////////////////////////////////////////////////////////
INT32 CRC_CACHE::LookupSet( UINT32 setIndex, Addr_t tag )
{
    // Get pointer to current set
    LINE_STATE *currSet = cache[ setIndex ];

    // Find Tag
    for(UINT32 way=0; way<assoc; way++) 
    {
        if( currSet[way].valid && (currSet[way].tag == tag) ) 
        {
            return way;
        }
    }

    // If not found, return -1
    return -1;
}

////////////////////////////////////////////////////////////////////////////////
//                                                                            //
// The function inspects the cache to see if the tag exists in the cache      //
// This function is used by the timing model for modeling latencies           //
//                                                                            //
////////////////////////////////////////////////////////////////////////////////
bool CRC_CACHE::CacheInspect( UINT32 tid, Addr_t PC, Addr_t paddr, UINT32 accessType ) 
{

    UINT32 setIndex = GetSetIndex( paddr );  // Get the set index
    Addr_t tag      = GetTag( paddr );       // Determine Cache Tag

    INT32 wayID     = LookupSet( setIndex, tag );

    // if wayID = -1, miss, else it is a hit
    return (wayID != -1);
}

////////////////////////////////////////////////////////////////////////////////
//                                                                            //
// This function is responsible for looking up and filling a line in the      //
// cache. If the result is a miss, the function consults the replacement      //
// policy for a victim candidate, inserts the missing line in the cache       //
// and consults the replacement policy for replacement state update. If       //
// the result was a cache hit, the replacement policy again is consulted      //
// to determine how to update the replacement state.                          //
//                                                                            //
////////////////////////////////////////////////////////////////////////////////
bool CRC_CACHE::LookupAndFillCache( UINT32 tid, Addr_t PC, Addr_t paddr, UINT32 accessType ) 
{

    LINE_STATE *currLine = NULL;

    // for modeling LRU
    ++mytimer;     
    cacheReplState->IncrementTimer();

    // manage stats for cache
    lookups[ accessType ][ tid ]++;

    // Process request
    bool  hit       = true;
    UINT32 setIndex = GetSetIndex( paddr );  // Get the set index
    Addr_t tag      = GetTag( paddr );       // Determine Cache Tag

    // Lookup the cache set to determine whether line is already in cache or not
    INT32 wayID     = LookupSet( setIndex, tag );

   
    if( wayID == -1 ) 
    {
        hit = false;

        // get victim line to replace (wayID = -1, then bypass)
        wayID     = GetVictimInSet( tid, setIndex, PC, paddr, accessType );

        if( wayID != -1 )
        {
            currLine  = &cache[ setIndex ][ wayID ];

            // Update the line state accordingly
            currLine->valid          = true;
            currLine->tag            = tag;
            currLine->dirty          = IS_STORE( accessType );
            currLine->sharing_dir    = (1<<tid);

            // Update Replacement State
            cacheReplState->UpdateReplacementState( setIndex, wayID, currLine, tid, PC, accessType, hit );
        }
        
        // Update Stats
        misses[ accessType ][ tid ]++;
    }
    else 
    {
        // get pointer to cache line we hit
        currLine         = &cache[ setIndex ][ wayID ];

        // Update the line state accordingly
        currLine->dirty         |= IS_STORE( accessType );
        currLine->sharing_dir   |= (1<<tid);

        // Update Replacement State
        if( accessType != ACCESS_WRITEBACK ) 
        {
            cacheReplState->UpdateReplacementState( setIndex, wayID, currLine, tid, PC, accessType, hit );
        }

        // Update Stats
        hits[ accessType ][ tid ]++;
    }        

    return hit;
}


////////////////////////////////////////////////////////////////////////////////
//                                                                            //
// This function is responsible for creating the cache replacement state      //
//                                                                            //
////////////////////////////////////////////////////////////////////////////////
void CRC_CACHE::InitCacheReplacementState()
{
    cacheReplState = new CACHE_REPLACEMENT_STATE( numsets, assoc, replPolicy );
}
